<?php
defined( 'ABSPATH' ) || exit;

/**
 * Elementor compatibility Class
 */
class Iconic_WSSV_Compat_Elementor {

	/**
	 * Init.
	 */
	public static function init() {
		if ( ! defined( 'ELEMENTOR_VERSION' ) ) {
			return;
		}

		add_action( 'pre_get_posts', array( __CLASS__, 'add_variations_to_autocomplete' ) );
		add_filter( 'posts_request', array( __CLASS__, 'add_variations_to_query' ), 10, 2 );
	}

	/**
	 * Add variations to autocomplete product search in admin.
	 *
	 * @param WP_Query $query WP_Query object.
	 */
	public static function add_variations_to_autocomplete( $query ) {
		if ( ! wp_doing_ajax() ) {
			return;
		}

		$action  = filter_input( INPUT_POST, 'action' );
		$actions = filter_input( INPUT_POST, 'actions' );

		if ( 'elementor_ajax' !== $action || empty( $actions ) ) {
			return;
		}

		$actions   = json_decode( $actions, true );
		$post_type = array_filter( (array) $query->get( 'post_type' ) );

		if ( empty( $actions ) ) {
			return;
		}

		if ( self::should_add_variations_to_query( $post_type, $actions ) ) {
			Iconic_WSSV_Query::add_variations_to_query( $query );
		} elseif ( isset( $actions['pro_panel_posts_control_filter_autocomplete'] ) ) {
			// Will be handled by add_variations_to_query().
			$query->set( 'iconic_ssv_add_variation_flag', true );
		}
	}

	/**
	 * Check if the variations should be add to the WP Query.
	 *
	 * @param array $post_types The post types already in the query.
	 * @param array $actions    The actions sent by Elementor.
	 * @return boolean
	 */
	protected static function should_add_variations_to_query( $post_types, $actions ) {
		if ( ! in_array( 'product', $post_types, true ) ) {
			return false;
		}

		$widgets_to_skip_adding_variations_to_query = array(
			'woocommerce-product-add-to-cart',
		);

		foreach ( $actions as $action ) {
			if ( empty( $action['action'] ) || 'render_widget' !== $action['action'] ) {
				continue;
			}

			if ( ! empty( $action['data']['data']['widgetType'] ) && in_array( $action['data']['data']['widgetType'], $widgets_to_skip_adding_variations_to_query, true ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Add variations to final SQL query generated by WP_Query
	 * when post type is 'any'.
	 *
	 * @param string   $sql   SQL query.
	 * @param WP_Query $query WP_Query object.
	 *
	 * @return string
	 */
	public static function add_variations_to_query( $sql, $query ) {
		$add_variations_flag = $query->get( 'iconic_ssv_add_variation_flag' );
		if ( ! $add_variations_flag ) {
			return $sql;
		}

		$post_type = $query->get( 'post_type' );
		if ( 'any' !== $post_type ) {
			return $sql;
		}

		/**
		 * Try to match: post_type = 'product'.
		 *
		 * Before of after `=` we can have n spaces and/or new lines.
		 */
		$sql_replaced = preg_replace( "/post_type[\s\S]*=[\s\S]*'product'/", "post_type IN ('product', 'product_variation')", $sql );

		if ( is_null( $sql_replaced ) ) {
			return $sql;
		}

		return $sql_replaced;
	}

}
