<?php
/**
 * Buy One Get One Free - Product Bundles by SomewhereWarm
 *
 * @see https://woocommerce.com/products/product-bundles/
 * @since 2.1.0
 * @package WC_BOGOF
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_BOGOF_Product_Bundles Class
 */
class WC_BOGOF_Product_Bundles {

	/**
	 * Retrun the minimun version required.
	 */
	public static function min_version_required() {
		return '6.3.0';
	}

	/**
	 * Returns the extension name.
	 */
	public static function extension_name() {
		return 'Product Bundles';
	}

	/**
	 * Checks the minimum version required.
	 */
	public static function check_min_version() {
		return defined( 'WC_PB_VERSION' ) ? version_compare( WC_PB_VERSION, static::min_version_required(), '>=' ) : false;
	}

	/**
	 * Init hooks
	 */
	public static function init() {
		add_filter( 'wc_bogof_load_conditions', array( __CLASS__, 'load_conditions' ) );
		add_filter( 'wc_bogof_buy_a_get_a_add_cart_item_data', array( __CLASS__, 'buy_a_get_a_add_cart_item_data' ) );
		add_action( 'wc_bogof_auto_add_to_cart', array( __CLASS__, 'auto_add_to_cart' ), 10, 2 );
		add_action( 'wc_bogof_after_cart_set_item_quantity', array( __CLASS__, 'after_cart_set_item_quantity' ), 10, 2 );
		add_action( 'wc_bogof_after_cart_remove_item', array( __CLASS__, 'after_cart_remove_item' ), 10, 2 );
		add_action( 'wc_bogof_cart_item_removed_from_session', array( __CLASS__, 'cart_item_removed_from_session' ), 10 );
		add_action( 'wc_bogof_after_set_cart_item_discount', array( __CLASS__, 'cart_item_discount_init' ), 10, 2 );
		add_action( 'wc_bogof_init_cart_item_discount', array( __CLASS__, 'cart_item_discount_init' ), 10, 2 );
		add_filter( 'wc_bogof_cart_rule_cheapest_free_cart_item_price', array( __CLASS__, 'cheapest_free_cart_item_price' ), 10, 2 );
		add_filter( 'wc_bogof_cart_template_item_price', array( __CLASS__, 'cart_template_item_price' ), 10, 3 );
		add_filter( 'wc_bogof_cart_template_total_save', array( __CLASS__, 'cart_template_total_save' ), 10, 2 );
		add_filter( 'woocommerce_bundled_item_hash', array( __CLASS__, 'bundled_item_hash' ), 10, 2 );
		add_filter( 'woocommerce_bundled_item_quantity', array( __CLASS__, 'bundled_item_quantity' ), 99999, 2 );
		add_filter( 'woocommerce_bundled_item_quantity_max', array( __CLASS__, 'bundled_item_quantity' ), 99999, 2 );

	}

	/**
	 * Add the All Product for WooCommerce Subsctiption condition.
	 *
	 * @param array $conditions Conditions array.
	 * @return array
	 */
	public static function load_conditions( $conditions ) {
		$conditions = is_array( $conditions ) ? $conditions : array();

		if ( ! class_exists( 'WC_BOGOF_Condition_Product_Bundles' ) ) {
			include_once dirname( WC_BOGOF_PLUGIN_FILE ) . '/includes/conditions/class-wc-bogof-condition-product-bundles.php';
		}

		$conditions[] = new WC_BOGOF_Condition_Product_Bundles();

		return $conditions;
	}

	/**
	 * Unset specific cart item keys.
	 *
	 * @param array $cart_item_data Cart item data.
	 */
	public static function buy_a_get_a_add_cart_item_data( $cart_item_data ) {
		if ( isset( $cart_item_data['bundled_item_id'] ) ) {
			unset(
				$cart_item_data['stamp'],
				$cart_item_data['bundled_items']
			);
		} elseif ( isset( $cart_item_data['bundled_items'] ) ) {
			$cart_item_data['bundled_items'] = [];
		}

		unset(
			$cart_item_data['bundled_by'],
			$cart_item_data['bundled_item_id']
		);

		return $cart_item_data;
	}

	/**
	 * Calls to add_to_cart function of product bundles.
	 *
	 * @param array $cart_item_data Cart item data.
	 */
	public static function auto_add_to_cart( $cart_item_data ) {
		if ( ! is_callable( [ 'WC_PB_Cart', 'instance' ] ) ) {
			return;
		}
		if ( ! is_callable( [ WC_PB_Cart::instance(), 'bundle_add_to_cart' ] ) ) {
			return;
		}

		WC_PB_Cart::instance()->bundle_add_to_cart(
			$cart_item_data['key'],
			$cart_item_data['product_id'],
			$cart_item_data['quantity'],
			$cart_item_data['variation_id'],
			$cart_item_data['variation'],
			$cart_item_data
		);
	}

	/**
	 * Calls to update_quantity_in_cart function of product bundles.
	 *
	 * @param string $cart_item_key Cart item key.
	 * @param int    $quantity Quantity.
	 */
	public static function after_cart_set_item_quantity( $cart_item_key, $quantity ) {
		if ( ! is_callable( [ 'WC_PB_Cart', 'instance' ] ) ) {
			return;
		}
		if ( ! is_callable( [ WC_PB_Cart::instance(), 'update_quantity_in_cart' ] ) ) {
			return;
		}

		WC_PB_Cart::instance()->update_quantity_in_cart( $cart_item_key, $quantity );
	}

	/**
	 * Calls to cart_item_remove function of product bundles.
	 *
	 * @param string $cart_item_key Cart item key.
	 * @param int    $quantity Quantity.
	 */
	public static function after_cart_remove_item( $cart_item_key, $quantity ) {
		if ( ! is_callable( [ 'WC_PB_Cart', 'instance' ] ) ) {
			return;
		}
		if ( ! is_callable( [ WC_PB_Cart::instance(), 'cart_item_remove' ] ) ) {
			return;
		}

		WC_PB_Cart::instance()->cart_item_remove( $cart_item_key, WC()->cart );
	}

	/**
	 * Recalculate the bundle base price.
	 *
	 * @param array                       $cart_item Cart item data.
	 * @param WC_BOGOF_Cart_Item_Discount $cart_discount The discount object.
	 */
	public static function cart_item_discount_init( $cart_item, $cart_discount ) {
		if ( ! ( wc_pb_is_bundle_container_cart_item( $cart_item ) ) ) {
			return;
		}

		$bundle_cart_keys = [];
		$bundled_items    = wc_pb_get_bundled_cart_items( $cart_item, WC()->cart->cart_contents );

		foreach ( $bundled_items as $bundled_item ) {

			if ( ! $bundled_item['data']->get_price() > 0 ) {
				continue;
			}

			$bundle_cart_keys[ $bundled_item['bundled_item_id'] ] = $bundled_item['key'];

			foreach ( $cart_discount->get_discounts() as $cart_rule_id => $discount_data ) {

				$bundled_item_discount = new WC_BOGOF_Cart_Item_Discount( $bundled_item );
				$bundled_item_discount->add_discount(
					$cart_rule_id,
					[
						'quantity' => $bundled_item['quantity'] / $cart_item['quantity'] * $discount_data->get_quantity(),
						'discount' => $discount_data->get_discount(),
					]
				);
				WC_BOGOF_Runtime_Meta::unset( $bundled_item['data'], 'discount' );
				WC_BOGOF_Runtime_Meta::set( $bundled_item['data'], 'discount', $bundled_item_discount );
			}
		}

		if ( WC_BOGOF_Cart::is_free_item( $cart_item ) ) {
			$bundle_cart_keys['bundle_container'] = $cart_item['key'];
			$cart_discount->add_extra_data( 'bundle_cart_keys', $bundle_cart_keys );
		}
	}

	/**
	 * Returns the cart item price for ordening.
	 *
	 * @since 5.5.5
	 *
	 * @param float $price The cart item price.
	 * @param array $cart_item Cart item data.
	 * @return float
	 */
	public static function cheapest_free_cart_item_price( $price, $cart_item ) {
		if ( ! ( wc_pb_is_bundle_container_cart_item( $cart_item ) ) ) {
			return $price;
		}

		$bundled_items = wc_pb_get_bundled_cart_items( $cart_item, WC()->cart->cart_contents );
		foreach ( $bundled_items as $bundled_item ) {
			$price += $bundled_item['data']->get_price() * absint( $bundled_item['quantity'] ) / absint( $cart_item['quantity'] );
		}

		return $price;
	}

	/**
	 * Filters the cart template item price.
	 *
	 * @since 5.5.5
	 *
	 * @param string $cart_item_price Price to display.
	 * @param array  $cart_item Cart item.
	 * @param string $cart_price Price without modifications.
	 * @return string
	 */
	public static function cart_template_item_price( $cart_item_price, $cart_item, $cart_price ) {

		if ( self::is_bundle_item_no_visible( $cart_item ) ) {

			$cart_item_price = '';

		} elseif ( wc_pb_is_bundle_container_cart_item( $cart_item ) ) {

			// Bundle container.

			if ( WC_Product_Bundle::group_mode_has( $cart_item['data']->get_group_mode(), 'aggregated_prices' ) ) {
				$base_price = self::get_cart_product_price( $cart_item['data'], WC_BOGOF_Runtime_Meta::get( $cart_item['data'], 'discount' )->get_base_price() );
				$price      = self::get_cart_product_price( $cart_item['data'] );

				$bundled_items = wc_pb_get_bundled_cart_items( $cart_item, WC()->cart->cart_contents );

				foreach ( $bundled_items as $bundled_item ) {
					if ( ! WC_BOGOF_Cart::is_valid_discount( $bundled_item ) ) {
						continue;
					}
					$base_price += self::get_cart_product_price( $bundled_item['data'], WC_BOGOF_Runtime_Meta::get( $bundled_item['data'], 'discount' )->get_base_price(), $bundled_item['quantity'] / $cart_item['quantity'] );
					$price      += self::get_cart_product_price( $bundled_item['data'], $bundled_item['data']->get_price(), $bundled_item['quantity'] / $cart_item['quantity'] );
				}

				if ( $price > 0 ) {

					$cart_item_price = wc_format_sale_price( $base_price, $price );

				} else {

					/**
					 * Filters the "Free!" text displayed for free items.
					 *
					 * @param string $cart_item_price Price to display.
					 * @param array  $cart_item Cart item.
					 */
					$cart_item_price = sprintf( '<span>%s</span>', apply_filters( 'wc_bogof_free_item_cart_price', __( 'Free!', 'woocommerce' ), $cart_item ) );
				}
			} else {

				$cart_item_price = $cart_price;
			}
		}

		return $cart_item_price;
	}

	/**
	 * Filters the cart template item price.
	 *
	 * @since 5.5.5
	 *
	 * @param float $total_save Total save.
	 * @param array $cart_item Cart item.
	 * @return float
	 */
	public static function cart_template_total_save( $total_save, $cart_item ) {
		if ( wc_pb_is_bundle_container_cart_item( $cart_item ) && WC_Product_Bundle::group_mode_has( $cart_item['data']->get_group_mode(), 'aggregated_prices' ) ) {

			$total_save    = WC_BOGOF_Runtime_Meta::get( $cart_item['data'], 'discount' )->get_amount();
			$bundled_items = wc_pb_get_bundled_cart_items( $cart_item, WC()->cart->cart_contents );

			foreach ( $bundled_items as $bundled_item ) {
				if ( ! WC_BOGOF_Cart::is_valid_discount( $bundled_item ) ) {
					continue;
				}

				$total_save += WC_BOGOF_Runtime_Meta::get( $bundled_item['data'], 'discount' )->get_amount();
			}

			$total_save = self::get_cart_product_price( $cart_item['data'], $total_save );

		} elseif ( self::is_bundle_item_no_visible( $cart_item ) ) {
			return 0;
		}

		return $total_save;
	}

	/**
	 * Returns the product price for display.
	 *
	 * @param WC_Product $product The product instance.
	 * @param float      $price Product price.
	 * @param int        $quantity Product quantity.
	 * @return float
	 */
	private static function get_cart_product_price( $product, $price = false, $quantity = 1 ) {
		return wc_bogof_get_cart_product_price(
			$product,
			[
				'price' => ( false === $price ? $product->get_price() : $price ),
				'qty'   => $quantity,
			]
		);
	}

	/**
	 * Is a bundle item no visible in cart.
	 *
	 * @param array $cart_item Cart item.
	 * @return bool
	 */
	private static function is_bundle_item_no_visible( $cart_item ) {
		if ( wc_pb_maybe_is_bundled_cart_item( $cart_item ) ) {

			// Bundle item.

			$bundle_container_item = wc_pb_get_bundled_cart_item_container( $cart_item );

			if ( $bundle_container_item ) {

				$bundled_item = $bundle_container_item['data']->get_bundled_item( $cart_item['bundled_item_id'] );

				if ( ! $bundled_item->is_price_visible( 'cart' ) || ! $bundled_item->is_priced_individually() ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Filters the bundle item hash.
	 *
	 * @since 5.5.5
	 * @param array             $hash The hash.
	 * @param WC_Product_Bundle $product Product instance.
	 */
	public static function bundled_item_hash( $hash, $product ) {
		$discount = WC_BOGOF_Runtime_Meta::get( $product, 'discount' );

		if ( $discount && $discount->get_extra_data( 'bundle_cart_keys' ) ) {
			$hash = array_merge(
				$hash,
				[
					'wc_bogof_gift' => $discount->get_extra_data( 'bundle_cart_keys' ),
				]
			);
		}

		return $hash;
	}

	/**
	 * Filter the bundle item quantity.
	 *
	 * @since 5.5.5
	 * @param int             $quantity Bundle item quantity.
	 * @param WC_Bundled_Item $bundled_item Bundle item instace.
	 */
	public static function bundled_item_quantity( $quantity, $bundled_item ) {
		$parent   = $bundled_item->get_bundle();
		$discount = $parent ? WC_BOGOF_Runtime_Meta::get( $parent, 'discount' ) : false;
		$data     = $discount && $discount->get_extra_data( 'bundle_cart_keys' ) ? $discount->get_extra_data( 'bundle_cart_keys' ) : false;
		if ( $data &&
			isset(
				$data[ $bundled_item->get_id() ],
				WC()->cart->cart_contents[ $data[ $bundled_item->get_id() ] ],
				WC()->cart->cart_contents[ $data['bundle_container'] ]
			)
		) {
			$bundled_cart_item = WC()->cart->cart_contents[ $data[ $bundled_item->get_id() ] ];
			$parent_cart_item  = WC()->cart->cart_contents[ $data['bundle_container'] ];

			return $bundled_cart_item['quantity'] / $parent_cart_item['quantity'];
		}

		return $quantity;
	}

	/**
	 * Cart item removed from session.
	 *
	 * @param array $cart_item Cart item data.
	 */
	public static function cart_item_removed_from_session( $cart_item ) {
		if ( wc_pb_is_bundle_container_cart_item( $cart_item ) ) {

			foreach ( $cart_item['bundled_items'] as $bundled_item_cart_key ) {
				unset( WC()->cart->cart_contents[ $bundled_item_cart_key ] );
			}
		}
	}
}
