<?php
/**
 * Quantity Rule Data data class. Encapsulate a quantity rule.
 *
 * @since 5.5.0
 * @package WC_BOGOF
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_BOGOF_Quantity_Rule_Data Class
 */
class WC_BOGOF_Data_Quantity_Rule {

	use WC_BOGOF_Data_Properties;

	/**
	 * Constructor.
	 *
	 * @param array $data Array of properties to set.
	 */
	public function __construct( $data = [] ) {
		$this->set_defaults();
		$this->set_key_to_prop_setter();
		$this->set_properties( $data );
	}

	/**
	 * Sets defaults.
	 */
	protected function set_defaults() {
		$this->data = [
			'cart_quantity' => 0,
			'free_quantity' => 0,
			'cart_limit'    => '',
			'discount'      => new WC_BOGOF_Data_Discount(),
		];
	}

	/**
	 * Sets the array of key value to map setter method with data keys.
	 */
	protected function set_key_to_prop_setter() {
		$this->key_to_prop_setter = [
			'discount'      => 'set_discount_amount',
			'discount_type' => 'set_discount_type',
		];
	}

	/**
	 * Sets the discount type.
	 *
	 * @param string $type Discount type.
	 */
	protected function set_discount_type( $type ) {
		$this->get_discount()->set_type( $type );
	}

	/**
	 * Sets the discount amount.
	 *
	 * @param float $amount Discount.
	 */
	protected function set_discount_amount( $amount ) {
		$this->get_discount()->set_amount( $amount );
	}

	/**
	 * Sets the cart quantity.
	 *
	 * @param int $quantity Quantity.
	 */
	public function set_cart_quantity( $quantity ) {
		$this->set_property( 'cart_quantity', absint( $quantity ) );
	}

	/**
	 * Sets the free quantity.
	 *
	 * @param int $quantity Quantity.
	 */
	public function set_free_quantity( $quantity ) {
		$this->set_property( 'free_quantity', absint( $quantity ) );
	}

	/**
	 * Sets the cart limit.
	 *
	 * @param int $limit Limit.
	 */
	public function set_cart_limit( $limit ) {
		if ( empty( $limit ) || ! absint( $limit ) > 0 ) {
			return;
		}
		$this->set_property( 'cart_limit', absint( $limit ) );
	}

	/**
	 * Sets the discount.
	 *
	 * @param WC_BOGOF_Data_Discount $discount Discount instance.
	 */
	public function set_discount( $discount ) {
		$this->set_property( 'discount', $discount );
	}

	/**
	 * Gets the cart quantity.
	 *
	 * @return int
	 */
	public function get_cart_quantity() {
		return $this->get_property( 'cart_quantity' );
	}

	/**
	 * Gets the free quantity.
	 *
	 * @return int
	 */
	public function get_free_quantity() {
		return $this->get_property( 'free_quantity' );
	}

	/**
	 * Gets the cart limit.
	 *
	 * @return int
	 */
	public function get_cart_limit() {
		return $this->get_property( 'cart_limit' );
	}

	/**
	 * Gets the discount instance.
	 *
	 * @return WC_BOGOF_Data_Discount
	 */
	public function get_discount() {
		return $this->get_property( 'discount' );
	}
}
