<?php
/**
 * Discount Data class. Encapsulate a discount.
 *
 * @since 5.5.3
 * @package WC_BOGOF
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_BOGOF_Data_Discount Class
 */
class WC_BOGOF_Data_Discount {

	use WC_BOGOF_Data_Properties;

	/**
	 * The allowed discount types.
	 */
	private const DISCOUNT_TYPES = [ 'percentage', 'fixed_product', 'fixed_price' ];

	/**
	 * Constructor.
	 *
	 * @param array $data Array of properties to set.
	 */
	public function __construct( $data = [] ) {
		$this->set_defaults();
		$this->set_properties( $data );
	}

	/**
	 * Sets defaults.
	 */
	protected function set_defaults() {
		$this->data = [
			'type'   => 'percentage',
			'amount' => 0,
		];
	}

	/**
	 * Sets the discount type.
	 *
	 * @param string $type Discount type.
	 */
	public function set_type( $type ) {
		if ( in_array( $type, self::DISCOUNT_TYPES, true ) ) {
			$this->set_property( 'type', $type );
		} else {
			/* translators: 1: Discount type */
			wc_doing_it_wrong( __FUNCTION__, sprintf( __( '"%1$s" discount type is not allowed.', 'wc-buy-one-get-one-free' ), $type ), '5.5.0' );
		}
	}

	/**
	 * Sets the discount.
	 *
	 * @param float $amount Discount.
	 */
	public function set_amount( $amount ) {
		$this->set_property( 'amount', floatval( $amount ) );
	}

	/**
	 * Gets the discount type.
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->get_property( 'type' );
	}

	/**
	 * Gets the discount.
	 *
	 * @return string
	 */
	public function get_amount() {
		return $this->get_property( 'amount' );
	}

	/**
	 * Returns the amount formatted.
	 */
	public function get_formatted_amount() {

		$formatted_amount = 'percentage' === $this->get_type() ?
			sprintf( '%s%%', $this->get_amount() ) :
			wc_price( $this->get_amount() );

		if ( 'fixed_price' !== $this->get_type() ) {
			// Translators: 1: discount amount.
			$formatted_amount = sprintf( __( '%1$s off', 'wc-buy-one-get-one-free' ), $formatted_amount );
		}

		return $formatted_amount;
	}
}
