<?php
/**
 * WooCommerce Buy One Get One Free cart template actions.
 *
 * @package WC_BOGOF
 */

use WC_BOGOF_Runtime_Meta as RuntimeMeta;

defined( 'ABSPATH' ) || exit;

/**
 * WC_BOGOF_Cart_Template Class
 */
class WC_BOGOF_Cart_Template {

	/**
	 * Init
	 */
	public static function init() {
		self::extend_store_api();
		self::init_hooks();
	}

	/**
	 * Init hooks
	 */
	public static function init_hooks() {
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'register_scripts' ), 20 );
		add_action( 'wp_footer', [ __CLASS__, 'enqueue_scripts' ], 5 );
		add_filter( 'woocommerce_cart_item_quantity', array( __CLASS__, 'cart_item_quantity' ), 9999, 3 );
		add_filter( 'woocommerce_cart_item_price', array( __CLASS__, 'cart_item_price' ), 99999, 2 );
		add_filter( 'woocommerce_cart_item_subtotal', array( __CLASS__, 'cart_item_subtotal' ), 99999, 2 );
		add_filter( 'woocommerce_coupon_discount_amount_html', array( __CLASS__, 'coupon_discount_amount_html' ), 9999, 2 );
		add_filter( 'woocommerce_store_api_product_quantity_maximum', array( __CLASS__, 'store_api_product_quantity_maximum' ), 9999, 3 );
		add_filter( 'woocommerce_store_api_product_quantity_minimum', array( __CLASS__, 'store_api_product_quantity_maximum' ), 9999, 3 );
		add_filter( 'woocommerce_store_api_product_quantity_editable', array( __CLASS__, 'store_api_product_quantity_editable' ), 9999, 3 );
	}

	/**
	 * Extend the Store API Cart Item schema.
	 */
	public static function extend_store_api() {
		if ( ! ( function_exists( 'woocommerce_store_api_register_endpoint_data' ) && class_exists( '\Automattic\WooCommerce\StoreApi\Schemas\V1\CartSchema' ) ) ) {
			return;
		}

		woocommerce_store_api_register_endpoint_data(
			[
				'endpoint'        => \Automattic\WooCommerce\StoreApi\Schemas\V1\CartItemSchema::IDENTIFIER,
				'namespace'       => strtolower( __CLASS__ ) . '_data',
				'data_callback'   => [ __CLASS__, 'store_api_cart_item_data' ],
				'schema_callback' => [ __CLASS__, 'store_api_cart_item_schema' ],
				'schema_type'     => ARRAY_A,
			]
		);
	}

	/**
	 * Register schema into cart endpoint.
	 *
	 * @return array Registered schema.
	 */
	public static function store_api_cart_item_schema() {
		return [
			'cart_item_class' => [
				'description' => __( '"Buy One Get One Free" cart item classes', 'wc-buy-one-get-one-free' ),
				'type'        => [ 'string', 'null' ],
				'context'     => array( 'view', 'edit' ),
				'readonly'    => true,
			],
		];
	}

	/**
	 * Adds data to the Store API cart route responses.
	 *
	 * @param array $cart_item Current cart item data.
	 * @return array Registered data.
	 */
	public static function store_api_cart_item_data( $cart_item ) {
		$classes      = [];
		$class_prefix = 'wc-bogof-block-item_';

		if ( WC_BOGOF_Cart::is_valid_free_item( $cart_item ) ) {
			$classes[] = "{$class_prefix}_gift";
		}

		if ( WC_BOGOF_Cart::is_valid_discount( $cart_item['data'] ) ) {
			if ( $cart_item['data']->get_price() > 0 ) {
				$classes[] = "{$class_prefix}_has_discount";
			} else {
				$classes[] = "{$class_prefix}_is_free";
			}
		}

		return [
			'cart_item_class' => implode( ' ', $classes ),
		];
	}

	/**
	 * Enqueue cart shortcode styles and the register cart block scripts.
	 */
	public static function register_scripts() {
		if ( is_cart() ) {
			// Cart shortcode.
			wp_enqueue_style( 'cart', plugins_url( 'assets/css/cart.css', WC_BOGOF_PLUGIN_FILE ), array(), WC_Buy_One_Get_One_Free::VERSION );
		}

		wp_register_style( 'wc-bogof-cart-block-template', plugins_url( 'assets/css/cart-block.css', WC_BOGOF_PLUGIN_FILE ), [], WC_Buy_One_Get_One_Free::VERSION );
		wp_register_script( 'wc-bogof-cart-block-template', plugins_url( 'assets/js/build/cart-block-template.js', WC_BOGOF_PLUGIN_FILE ), [ 'wc-blocks-checkout' ], WC_Buy_One_Get_One_Free::VERSION, true );
	}

	/**
	 * Enqueue cart cart block scripts.
	 */
	public static function enqueue_scripts() {
		if ( wp_script_is( 'wc-blocks-checkout', 'registered' ) && ( wp_script_is( 'wc-mini-cart-block-frontend' ) || wp_script_is( 'wc-cart-block-frontend' ) ) ) {
			wp_enqueue_style( 'wc-bogof-cart-block-template' );
			wp_enqueue_script( 'wc-bogof-cart-block-template' );
		}
	}

	/**
	 * Quantity of free items have not be able updated
	 *
	 * @param int    $product_quantity Product quantity.
	 * @param string $cart_item_key Cart item key.
	 * @param array  $cart_item Cart item.
	 * @return string
	 */
	public static function cart_item_quantity( $product_quantity, $cart_item_key, $cart_item = false ) {
		if ( ! $cart_item ) {
			$cart_item = WC()->cart->get_cart_item( $cart_item_key );
		}

		if ( $cart_item && WC_BOGOF_Cart::is_valid_free_item( $cart_item ) ) {
			$product_quantity = sprintf( '<span>%s</span> <input type="hidden" name="cart[%s][qty]" value="%s" />', $cart_item['quantity'], $cart_item_key, $cart_item['quantity'] );
		}
		return $product_quantity;
	}

	/**
	 * Cart item price.
	 *
	 * @param string $cart_price Price to display.
	 * @param array  $cart_item Cart item.
	 */
	public static function cart_item_price( $cart_price, $cart_item ) {
		if ( ! ( isset( $cart_item['data'] ) && WC_BOGOF_Cart::is_valid_discount( $cart_item['data'] ) ) ) {
			return $cart_price;
		}

		$price = $cart_item['data']->get_price();

		if ( $price > 0 ) {

			$cart_item_price = wc_format_sale_price(
				wc_bogof_get_cart_product_price(
					$cart_item['data'],
					array(
						'price' => RuntimeMeta::get( $cart_item['data'], 'discount' )->get_base_price(),
						'qty'   => 1,
					)
				),
				wc_bogof_get_cart_product_price(
					$cart_item['data'],
					array(
						'price' => $price,
						'qty'   => 1,
					)
				)
			);
		} else {

			/**
			 * Filters the "Free!" text displayed for free items.
			 *
			 * @param string $cart_price Price to display.
			 * @param array  $cart_item Cart item.
			 */
			$cart_item_price = sprintf( '<span>%s</span>', apply_filters( 'wc_bogof_free_item_cart_price', __( 'Free!', 'woocommerce' ), $cart_item ) );
		}

		/**
		 * Filters the cart template item price.
		 *
		 * @since 5.5.5
		 *
		 * @param string $cart_price Price to display.
		 * @param array  $cart_item Cart item.
		 * @param string $cart_price Price without modifications.
		 */
		return apply_filters( 'wc_bogof_cart_template_item_price', $cart_item_price, $cart_item, $cart_price );
	}

	/**
	 * Cart item subtotal. For BOGO offers display the discount.
	 *
	 * @param string $cart_subtotal Subtotal to display.
	 * @param array  $cart_item Cart item.
	 */
	public static function cart_item_subtotal( $cart_subtotal, $cart_item ) {
		if ( ! WC_BOGOF_Cart::is_valid_discount( $cart_item ) || is_checkout() ) {
			return $cart_subtotal;
		}

		$item_discount = RuntimeMeta::get( $cart_item['data'], 'discount' );

		if ( $item_discount->get_free_quantity() === absint( $cart_item['quantity'] ) ) {
			return $cart_subtotal;
		}

		/**
		 * Filters the cart template total save
		 *
		 * @since 5.5.5
		 *
		 * @param float $total_save Total save,
		 * @param array  $cart_item Cart item.
		 */
		$total_save = apply_filters(
			'wc_bogof_cart_template_total_save',
			wc_bogof_get_cart_product_price( $cart_item['data'], [ 'price' => $item_discount->get_amount() ] ),
			$cart_item
		);

		if ( $total_save > 0 ) {
			/**
			 * Filters the subtotal "Save" badge text.
			 *
			 * @since 5.5.0
			 *
			 * @param string                      $save_text The content that will be printed.
			 * @param string                      $total_save The total save.
			 * @param WC_BOGOF_Cart_Item_Discount $item_discount Discount instance.
			 * @param array                       $cart_item Cart item array.
			 */
			$save_text = apply_filters(
				'wc_bogof_cart_subtotal_save_badge',
				sprintf(
					'%s %s&nbsp;&times;&nbsp;%s',
					__( 'Save', 'woocommerce' ),
					$item_discount->get_free_quantity(),
					wc_price( $total_save / $item_discount->get_free_quantity() )
				),
				$total_save,
				$item_discount,
				$cart_item
			);

			if ( $save_text ) {
				$cart_subtotal .= sprintf( '<span class="wc-bogof-cart-subtotal-sale-badge">%s</span>', $save_text );
			}
		}

		return $cart_subtotal;

	}

	/**
	 * Coupon amount HTML.
	 *
	 * @param string    $discount_amount_html Coupon amount HTML.
	 * @param WC_Coupon $coupon Coupon object.
	 * @return string
	 */
	public static function coupon_discount_amount_html( $discount_amount_html, $coupon ) {

		$bogo_coupon = WC_BOGOF_Coupon::get( $coupon->get_code() );

		if ( $bogo_coupon->is_in_rule() &&
			wc_bogof_coupon_is_empty( $coupon ) &&
			apply_filters( 'wc_bogof_hide_empty_coupon_amount_html', true, $coupon )
		) {
			// Do not display 0.00 on BOGO coupons.
			$discount_amount_html = '';
		}

		return $discount_amount_html;
	}

	/**
	 * Disable the input quantity field on cart and checkout blocks.
	 *
	 * @param mixed       $value The value being filtered.
	 * @param \WC_Product $product The product object.
	 * @param array       $cart_item The cart item if the product exists in the cart, or null.
	 * @return mixed
	 */
	public static function store_api_product_quantity_maximum( $value, $product, $cart_item ) {

		if ( isset( $cart_item['quantity'] ) && WC_BOGOF_Cart::is_free_item( $cart_item ) ) {
			$value = $cart_item['quantity'];
		}
		return $value;
	}

	/**
	 * Hides the input quantity field on cart and checkout blocks if the item quantity is 1.
	 *
	 * @param mixed       $value The value being filtered.
	 * @param \WC_Product $product The product object.
	 * @param array       $cart_item The cart item if the product exists in the cart, or null.
	 * @return mixed
	 */
	public static function store_api_product_quantity_editable( $value, $product, $cart_item ) {
		if ( isset( $cart_item['quantity'] ) && WC_BOGOF_Cart::is_free_item( $cart_item ) && 1 === absint( $cart_item['quantity'] ) ) {
			return false;
		}

		return $value;
	}
}
