<?php
/**
 * Buy One Get One Free Discount calculation.
 *
 * @package WC_BOGOF
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_BOGOF_Choose_Discount Class
 */
class WC_BOGOF_Cart_Item_Discount {

	/**
	 * Array of properties.
	 *
	 * @var array
	 */
	protected $data;

	/**
	 * WC_BOGOF_Cart_Item_Discount Constructor.
	 *
	 * @param array $cart_item_data Cart item data.
	 * @param array $discounts Discounts as a cart rule ID => Discount data pair array.
	 */
	public function __construct( $cart_item_data = null, $discounts = null ) {
		$this->set_defaults();

		if ( ! ( is_array( $cart_item_data ) && isset( $cart_item_data['data'] ) && isset( $cart_item_data['quantity'] ) && is_a( $cart_item_data['data'], 'WC_Product' ) ) ) {
			return;
		}

		$product = $cart_item_data['data'];

		$this->set_product_id( $product->get_id() );
		$this->set_base_price(
			'yes' === get_option( 'wc_bogof_base_regular_price', 'no' ) ?
			$product->get_regular_price() :
			$product->get_price()
		);
		$this->set_cart_quantity( $cart_item_data['quantity'] );

		if ( is_array( $discounts ) ) {
			foreach ( $discounts as $cart_rule_id => $discount_data ) {
				$this->add_discount( $cart_rule_id, $discount_data );
			}
		}
	}

	/**
	 * Sets defaults.
	 */
	protected function set_defaults() {
		$this->data = [
			'discounts'     => [],
			'base_price'    => 0,
			'cart_quantity' => 0,
			'product_id'    => 0,
			'cache'         => [],
		];
	}

	/**
	 * Sets a single property.
	 *
	 * @param string $prop Property.
	 * @param mixed  $value Value.
	 */
	protected function set_property( $prop, $value ) {
		if ( isset( $this->data[ $prop ] ) ) {
			$this->data[ $prop ] = $value;
			$this->data['cache'] = [];
		}
	}

	/**
	 * Sets cache.
	 *
	 * @param string $key Cache key.
	 * @param mixed  $value Value.
	 */
	protected function set_cache( $key, $value ) {
		$this->data['cache'][ $key ] = $value;
	}

	/**
	 * Gets cache by key.
	 *
	 * @param string $key Cache key.
	 */
	protected function get_cache( $key ) {
		return isset( $this->data['cache'][ $key ] ) ? $this->data['cache'][ $key ] : false;
	}

	/**
	 * Set product ID.
	 *
	 * @param int $product_id Product ID.
	 */
	public function set_product_id( $product_id ) {
		$this->set_property( 'product_id', absint( $product_id ) );
	}

	/**
	 * Returns the product ID.
	 *
	 * @return int
	 */
	public function get_product_id() {
		return $this->data['product_id'];
	}

	/**
	 * Set base price.
	 *
	 * @param float $base_price Base price.
	 */
	public function set_base_price( $base_price ) {
		$this->set_property( 'base_price', floatval( $base_price ) );
	}

	/**
	 * Returns the base price of the cart item.
	 *
	 * @return float
	 */
	public function get_base_price() {
		return $this->data['base_price'];
	}

	/**
	 * Set cart quantity.
	 *
	 * @param int $cart_quantity Base price.
	 */
	public function set_cart_quantity( $cart_quantity ) {
		$this->set_property( 'cart_quantity', absint( $cart_quantity ) );
	}

	/**
	 * Returns cart quantity.
	 */
	public function get_cart_quantity() {
		return $this->data['cart_quantity'];
	}

	/**
	 * Adds a discount.
	 *
	 * @param mixed                             $cart_rule_id Rule ID.
	 * @param WC_BOGOF_Data_Item_Discount|array $discount Item discount or Item discount data as array.
	 */
	public function add_discount( $cart_rule_id, $discount ) {

		if ( is_array( $discount ) ) {
			$discount = new WC_BOGOF_Data_Item_Discount( $discount );
		}

		if ( ! is_a( $discount, 'WC_BOGOF_Data_Item_Discount' ) ) {
			return;
		}

		if ( $discount->get_quantity() < 1 ) {
			$this->remove_discount( $cart_rule_id );
		} else {
			$this->data['discounts'][ $cart_rule_id ] = $discount;
			$this->data['cache']                      = [];
		}
	}

	/**
	 * Remove a discount.
	 *
	 * @param mixed $cart_rule_id Rule ID.
	 */
	public function remove_discount( $cart_rule_id ) {
		unset( $this->data['discounts'][ $cart_rule_id ] );
		$this->data['cache'] = [];
	}

	/**
	 * Returns the discounts as a key => discount pair array
	 *
	 * @return array
	 */
	public function get_discounts() {
		return $this->data['discounts'];
	}

	/**
	 * Add extra data.
	 *
	 * @param string $key Extra data key.
	 * @param mixed  $value Value.
	 */
	public function add_extra_data( $key, $value ) {
		$this->data[ "_{$key}" ] = $value;
	}

	/**
	 * Remove extra data.
	 *
	 * @param string $key Extra data key to remove.
	 */
	public function remove_extra_data( $key ) {
		unset( $this->data[ "_{$key}" ] );
	}

	/**
	 * Get extra data.
	 *
	 * @param string $key Extra data key to get.
	 * @return mixed
	 */
	public function get_extra_data( $key ) {
		return isset( $this->data[ "_{$key}" ] ) ? $this->data[ "_{$key}" ] : false;
	}


	/**
	 * Returns the cart rule IDs.
	 *
	 * @return array
	 */
	public function get_cart_rule_ids() {
		return array_keys( $this->get_discounts() );
	}

	/**
	 * Check a cart rule ID or an array of cart rule IDs.
	 *
	 * @param array|string $cart_rule_id Cart rule ID.
	 * @return bool
	 */
	public function has_cart_rule( $cart_rule_id ) {
		$cart_rule_id = is_array( $cart_rule_id ) ? $cart_rule_id : array( $cart_rule_id );
		return wc_bogof_in_array_intersect( array_keys( $this->get_discounts() ), $cart_rule_id );
	}

	/**
	 * Returns if there are free items.
	 *
	 * @return bool
	 */
	public function has_discount() {
		return $this->get_free_quantity() > 0;
	}

	/**
	 * Returns total free quantity.
	 *
	 * @return int
	 */
	public function get_free_quantity() {
		return array_sum(
			wc_list_pluck(
				$this->get_discounts(),
				'get_quantity'
			)
		);
	}

	/**
	 * Returns the discount amount.
	 *
	 * @return float
	 */
	public function get_amount() {

		$cache_key = __FUNCTION__;

		if ( $this->get_cache( $cache_key ) && is_numeric( $this->get_cache( $cache_key ) ) ) {
			return $this->get_cache( $cache_key );
		}

		$amount      = 0;
		$product     = wc_get_product( $this->get_product_id() );
		$cart_price  = wc_bogof_get_cart_product_price( $product, [ 'price' => $this->get_base_price() ] );
		$fixed_price = [];

		foreach ( $this->get_discounts() as $item ) {

			switch ( $item->get_discount()->get_type() ) {
				case 'percentage':
					$amount += $this->get_base_price() * $item->get_quantity() * $item->get_discount()->get_amount() / 100;
					break;
				case 'fixed_product':
					$amount += ( $item->get_discount()->get_amount() / $cart_price ) * $item->get_quantity() * $this->get_base_price();
					break;
				case 'fixed_price':
					// There is a fixed sale price. Store it to apply the lowest.
					array_push( $fixed_price, $item->get_discount()->get_amount() );
			}
		}

		if ( $fixed_price ) {
			$sale_price = min( array_merge( $fixed_price, [ $this->get_base_price() ] ) );
			$amount     = ( $this->get_base_price() - $sale_price ) * $this->get_free_quantity();
		}

		if ( $amount > $this->get_base_price() * $this->get_free_quantity() ) {
			$amount = $this->get_base_price() * $this->get_free_quantity();
		}

		$this->set_cache( $cache_key, $amount );

		return $amount;
	}

	/**
	 * Returns the sale price of the cart item or false if there is no discount.
	 *
	 * @return float|bool
	 */
	public function get_sale_price() {

		$cache_key = __FUNCTION__;

		if ( $this->get_cache( $cache_key ) && is_numeric( $this->get_cache( $cache_key ) ) ) {
			return $this->get_cache( $cache_key );
		}

		$discount   = $this->get_amount();
		$sale_price = false;

		if ( $discount > 0 ) {

			$base_price = $this->get_base_price() * $this->get_cart_quantity();
			$sale_price = ( $base_price - $discount ) / $this->get_cart_quantity();

			/**
			 * Filters the cart item dicount sale price.
			 *
			 * @param float                       $sale_price Sale price.
			 * @param WC_BOGOF_Cart_Item_Discount $item_discount Item discount instance.
			 */
			$sale_price = apply_filters( 'wc_bogof_cart_item_discount_sale_price', $sale_price, $this );
		}

		$this->set_cache( $cache_key, $sale_price );

		return $sale_price;
	}

	/*
	|--------------------------------------------------------------------------
	| Deprecated
	|--------------------------------------------------------------------------
	*/

	/**
	 * Set free quantity.
	 *
	 * @deprecated 5.5.3
	 * @param mixed $cart_rule_id Rule ID.
	 * @param int   $free_qty Free quantity.
	 */
	public function set_free_quantity( $cart_rule_id, $free_qty ) {
		wc_deprecated_function( __CLASS__ . '::set_free_quantity', '5.5.3', __CLASS__ . '::add_discount' );

		$cart_rule = wc_bogof_cart_rules()->get( $cart_rule_id );
		if ( ! $cart_rule ) {
			return;
		}

		$quantity_rule = $cart_rule->get_quantity_rule();
		if ( ! $quantity_rule ) {
			return;
		}

		$this->add_discount(
			$cart_rule_id,
			[
				'quantity' => $free_qty,
				'discount' => wc_bogof_cart_rules()->get( $cart_rule_id )->get_quantity_rule()->get_discount(),
			]
		);
	}

	/**
	 * Remove free quantity.
	 *
	 * @deprecated 5.5.3
	 * @param mixed $cart_rule_id Rule ID.
	 */
	public function remove_free_quantity( $cart_rule_id ) {
		wc_deprecated_function( __CLASS__ . '::remove_free_quantity', '5.5.3', __CLASS__ . '::remove_discount' );
		$this->remove_discount( $cart_rule_id );
	}

	/**
	 * Retuns the rule array
	 *
	 * @deprecated 5.5.3
	 * @return array
	 */
	public function get_rules() {
		wc_deprecated_function( __CLASS__ . '::get_rules', '5.5.3', __CLASS__ . '::get_discounts' );
		return wc_list_pluck( $this->get_discounts(), 'get_quantity' );
	}
}
