<?php

/**
 * Plugin Name: BWL Advanced FAQ Manager
 * Plugin URI: https://1.envato.market/baf-wp
 * Description: A WordPress plugin designed for managing unlimited number of FAQs and display on your website.
 * Author: Mahbub Alam Khan
 * Version: 2.1.3
 * Author URI: https://codecanyon.net/user/xenioushk
 * WP Requires at least: 6.0+
 * Text Domain: bwl-adv-faq
 * Domain Path: /lang/
 *
 * @package BWL Advanced FAQ Manager
 * @author Mahbub Alam Khan
 * @license GPL-2.0+
 * @link https://codecanyon.net/user/xenioushk
 * @copyright 2025 BlueWindLab
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit();
}

if ( file_exists( __DIR__ . '/vendor/autoload.php' ) ) {
    require_once __DIR__ . '/vendor/autoload.php';
}

use BwlFaqManager\Base\Activate;
use BwlFaqManager\Base\Deactivate;
use BwlFaqManager\Api\Db\BafDbSettings;
use BwlFaqManager\Base\BafAdminHelpers;
use BwlFaqManager\Api\Cpt\BafCustomPostType;
use BwlFaqManager\Api\Noticebox\BafAdminNotice;
use BwlFaqManager\Api\ViewsTracker\BafViewsTracker;
use BwlFaqManager\Api\KeyWordsTracker\BafSearchKeywordsTracker;

if ( ! class_exists( 'BWL_Advanced_Faq_Manager' ) ) {
    require_once __DIR__ . '/includes/BafConstants.php';

    /**
     * Main class for the BWL Advanced FAQ Manager plugin.
     *
     * This class handles the initialization and setup of the plugin, including
     * enqueuing scripts, loading translation files, and including necessary files.
     *
     * @package BWL Advanced FAQ Manager
     * @since 1.0.0
     */
    class BWL_Advanced_Faq_Manager {


        /**
         * Constructor for the BWL_Advanced_Faq_Manager class.
         *
         * This sets up the plugin by defining constants, including necessary files,
         * and adding actions for enqueuing scripts and loading translation files.
         */
        public function __construct() {
            define( 'BWL_BAF_PLUGIN_VERSION', '2.1.3' );
            define( 'BWL_BAF_PLUGIN_UPDATER_SLUG', plugin_basename( __FILE__ ) );
            define( 'BWL_BAF_PLUGIN_PATH', __DIR__ );
            define( 'BWL_BAF_INSTALLATION_TAG', 'baf_installation_' . str_replace( '.', '_', BWL_BAF_PLUGIN_VERSION ) );

            define( 'BWL_BAF_PLUGIN_CONTROLLER_PATH', __DIR__ . 'includes/Controllers/' );
            define( 'BWL_BAF_PLUGIN_VIEWS_PATH', __DIR__ . '/includes/Views/' );
            $this->include_files();

            add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
            add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_scripts' ] );
            add_action( 'plugins_loaded', [ $this, 'load_translation_file' ] );
        }

        /**
         * Include front-end styles and scripts
         *
         * @since 1.0.0
         */
        public function enqueue_scripts() {

            $options = get_option( 'bwl_advanced_faq_options' );

            wp_enqueue_style( 'baf-frontend', plugins_url( 'assets/styles/frontend.css', __FILE__ ), [], BWL_BAF_PLUGIN_VERSION );

            if ( is_rtl() ) {

                wp_enqueue_style( 'baf-frontend-rtl', plugins_url( 'assets/styles/frontend_rtl.css', __FILE__ ), [], BWL_BAF_PLUGIN_VERSION );
            }

            if ( isset( $options['bwl_advanced_fa_status'] ) && $options['bwl_advanced_fa_status'] === 'on' ) {

                wp_enqueue_style(
                    'font-awesome',
                    plugins_url( 'libs/font-awesome/font-awesome.min.css', __FILE__ ),
                    [],
                    BWL_BAF_PLUGIN_VERSION
                );

                wp_enqueue_style(
                    'font-awesome-shims',
                    plugins_url( 'libs/font-awesome/v4-shims.min.css', __FILE__ ),
                    [],
                    BWL_BAF_PLUGIN_VERSION
                );
            }

            wp_enqueue_script(
                'baf-frontend',
                plugins_url( 'assets/scripts/frontend.js', __FILE__ ),
                [ 'jquery' ],
                BWL_BAF_PLUGIN_VERSION,
                true
            );

            // Variable access Rules: BafFrontendData.ajaxurl

            wp_localize_script(
                'baf-frontend',
                'BafFrontendData', // javascript end variable.
                BafHelpers::localize_data()
            );
        }

        /**
         * Include back-end styles and scripts
         *
         * @since 1.0.0
         */
        public function enqueue_admin_scripts() {

            global $pagenow;

            // Load admin styles & scripts.
            wp_enqueue_style(
                'baf-admin',
                plugins_url( 'assets/styles/admin.css', __FILE__ ),
                [ 'wp-color-picker' ],
                BWL_BAF_PLUGIN_VERSION
            );

            if ( is_rtl() ) {
                wp_enqueue_style(
                    'baf-admin-rtl',
                    plugins_url( 'assets/styles/admin_rtl.css', __FILE__ ),
                    [],
                    BWL_BAF_PLUGIN_VERSION
                );
            }

            // TinyMCE Editor Style.
            wp_register_style(
                'bwl-advanced-faq-editor-style',
                plugins_url( 'libs/tinymce/styles/bwl-advanced-faq-editor.css', __FILE__ ),
                [],
                BWL_BAF_PLUGIN_VERSION
            );
            wp_register_style(
                'bwl-advanced-faq-multiple-select',
                plugins_url( 'libs/multiple-select/styles/multiple-select.css', __FILE__ ),
                [],
                BWL_BAF_PLUGIN_VERSION
            );
            wp_register_script(
                'bwl-advanced-faq-multiple-select',
                plugins_url( 'libs/multiple-select/scripts/jquery.multiple.select.js', __FILE__ ),
                [ 'jquery', 'jquery-ui-core', 'jquery-ui-draggable', 'jquery-ui-droppable' ],
                BWL_BAF_PLUGIN_VERSION,
                true
            );

            // Chart JS.
            wp_register_script(
                'baf-chart-js',
                plugins_url( 'libs/chart-js/scripts/chart.js', __FILE__ ),
                [ 'jquery' ],
                BWL_BAF_PLUGIN_VERSION,
                true
            );

            // Enqueue FAQ Admin Script & Style.
            wp_enqueue_style( 'bwl-advanced-faq-editor-style' ); // TinyMCE Editor Overlay.
            wp_enqueue_style( 'bwl-advanced-faq-multiple-select' ); // Enqueue Multiselect Style.
            wp_enqueue_script( 'bwl-advanced-faq-multiple-select' ); // Enqueue Multiselect Script.

            if ( $pagenow === 'edit.php' && isset( $_GET['post_type'] ) && $_GET['post_type'] === 'bwl_advanced_faq' ) {
                wp_enqueue_script( 'baf-chart-js' );
            }

            wp_enqueue_script(
                'baf-admin',
                plugins_url( 'assets/scripts/admin.js', __FILE__ ),
                [ 'jquery', 'wp-color-picker', 'jquery-ui-core', 'jquery-ui-draggable', 'jquery-ui-droppable', 'jquery-ui-sortable' ],
                BWL_BAF_PLUGIN_VERSION,
                true
            );

            // Variable access Rules: BafFrontendData.ajaxurl
            wp_localize_script(
                'baf-admin',
                'BafAdminData',
                BafAdminHelpers::localize_data()
            );
        }

        /**
         * Include plugin required files
         *
         * @since 1.0.0
         */
        public function include_files() {

            // Commen Functions.
            require_once __DIR__ . '/includes/BafHelpers.php';

            // Upgrade Custom DataBase Tables
            new BafDbSettings();

            // Custom Post Type
            new BafCustomPostType();

            // View Tracker
            new BafViewsTracker();

            // Search Keywords Tracker
            new BafSearchKeywordsTracker();

            if ( is_admin() ) {

                // Load plugin versioning informations
                require_once __DIR__ . '/includes/version-manager.php';

                // Load Custom shrotcode editor panel
                require_once __DIR__ . '/includes/tinymce/baf_tiny_mce_config.php';

                // Options Panel
                require_once __DIR__ . '/includes/settings/options_panel.php'; // Load plugins options panel.
                require_once __DIR__ . '/includes/settings/options_panel_submenus.php'; // Load Welcome page.

                // Auto Updater Scripts
                require_once __DIR__ . '/includes/autoupdater/WpAutoUpdater.php';
                require_once __DIR__ . '/includes/autoupdater/installer.php';
                require_once __DIR__ . '/includes/autoupdater/updater.php';
                require_once __DIR__ . '/includes/autoupdater/baf_update_notifier.php';
                require_once __DIR__ . '/includes/autoupdater/plugin_purchase_verify.php';

                // Admin Notice
                new BafAdminNotice();
            }
        }

        /**
         * Include plugin translation file
         *
         * @since 1.0.0
         */
        public function load_translation_file() {
            load_plugin_textdomain( 'bwl-adv-faq', false, dirname( plugin_basename( __FILE__ ) ) . '/lang/' );
        }
    }

    /**
     * Include plugin widgets
     *
     * @since 1.0.0
     */

    $baf_widgets = [ 'faqs', 'categories', 'topics' ];

    foreach ( $baf_widgets as $widget_key => $widget_page ) :
        require_once __DIR__ . '/includes/widgets/baf_' . $widget_page . '.php';
    endforeach;

    /**
     * Initialize the plugin
     *
     * @since 1.0.0
     */
    function init_bwl_advanced_faq_manager()
    { // phpcs:ignore
        new BWL_Advanced_Faq_Manager();
    }

    add_action( 'init', 'init_bwl_advanced_faq_manager' );

    /**
     * Plugin activation hook
     *
     * @since 1.0.0
     */
    function baf_active_plugin()
    { // phpcs:ignore
        Activate::activate();
    }

    register_activation_hook( __FILE__, 'baf_active_plugin' );

    /**
     * Plugin deactivation hook
     *
     * @since 1.0.0
     */
    function baf_deactive_plugin() {
        Deactivate::deactivate();
    }
    register_activation_hook( __FILE__, 'baf_deactive_plugin' );

    /**
     * Register all required classes for the plugin.
     *
     * @since 1.0.0
     */

    if ( class_exists( 'BwlFaqManager\\Init' ) ) {

        BwlFaqManager\Init::register_services();
    }
}
